/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/

import { sep } from '../../../../../base/common/path.js';
import { raceCancellationError } from '../../../../../base/common/async.js';
import { CancellationToken } from '../../../../../base/common/cancellation.js';
import { Codicon } from '../../../../../base/common/codicons.js';
import { AsyncEmitter, Emitter, Event } from '../../../../../base/common/event.js';
import { combinedDisposable, Disposable, DisposableMap, DisposableStore, IDisposable, toDisposable } from '../../../../../base/common/lifecycle.js';
import { ResourceMap } from '../../../../../base/common/map.js';
import { Schemas } from '../../../../../base/common/network.js';
import * as resources from '../../../../../base/common/resources.js';
import { ThemeIcon } from '../../../../../base/common/themables.js';
import { URI, UriComponents } from '../../../../../base/common/uri.js';
import { generateUuid } from '../../../../../base/common/uuid.js';
import { localize, localize2 } from '../../../../../nls.js';
import { Action2, IMenuService, MenuId, MenuItemAction, MenuRegistry, registerAction2 } from '../../../../../platform/actions/common/actions.js';
import { ContextKeyExpr, IContextKey, IContextKeyService } from '../../../../../platform/contextkey/common/contextkey.js';
import { IRelaxedExtensionDescription } from '../../../../../platform/extensions/common/extensions.js';
import { InstantiationType, registerSingleton } from '../../../../../platform/instantiation/common/extensions.js';
import { ServicesAccessor } from '../../../../../platform/instantiation/common/instantiation.js';
import { ILabelService } from '../../../../../platform/label/common/label.js';
import { ILogService } from '../../../../../platform/log/common/log.js';
import { isDark } from '../../../../../platform/theme/common/theme.js';
import { IThemeService } from '../../../../../platform/theme/common/themeService.js';
import { IEditorService } from '../../../../services/editor/common/editorService.js';
import { IExtensionService, isProposedApiEnabled } from '../../../../services/extensions/common/extensions.js';
import { ExtensionsRegistry } from '../../../../services/extensions/common/extensionsRegistry.js';
import { ChatEditorInput } from '../widgetHosts/editor/chatEditorInput.js';
import { IChatAgentAttachmentCapabilities, IChatAgentData, IChatAgentService } from '../../common/participants/chatAgents.js';
import { ChatContextKeys } from '../../common/actions/chatContextKeys.js';
import { IChatSession, IChatSessionContentProvider, IChatSessionItem, IChatSessionItemProvider, IChatSessionOptionsWillNotifyExtensionEvent, IChatSessionProviderOptionGroup, IChatSessionProviderOptionItem, IChatSessionsExtensionPoint, IChatSessionsService, isSessionInProgressStatus } from '../../common/chatSessionsService.js';
import { ChatAgentLocation, ChatModeKind } from '../../common/constants.js';
import { CHAT_CATEGORY } from '../actions/chatActions.js';
import { IChatEditorOptions } from '../widgetHosts/editor/chatEditor.js';
import { IChatModel } from '../../common/model/chatModel.js';
import { IChatService, IChatToolInvocation } from '../../common/chatService/chatService.js';
import { autorun, autorunIterableDelta, observableFromEvent, observableSignalFromEvent } from '../../../../../base/common/observable.js';
import { IChatRequestVariableEntry } from '../../common/attachments/chatVariableEntries.js';
import { renderAsPlaintext } from '../../../../../base/browser/markdownRenderer.js';
import { IMarkdownString } from '../../../../../base/common/htmlContent.js';
import { IViewsService } from '../../../../services/views/common/viewsService.js';
import { ChatViewId } from '../chat.js';
import { ChatViewPane } from '../widgetHosts/viewPane/chatViewPane.js';
import { AgentSessionProviders, getAgentSessionProviderName } from '../agentSessions/agentSessions.js';
import { BugIndicatingError } from '../../../../../base/common/errors.js';
import { IEditorGroupsService } from '../../../../services/editor/common/editorGroupsService.js';
import { LocalChatSessionUri } from '../../common/model/chatUri.js';
import { assertNever } from '../../../../../base/common/assert.js';
import { ICommandService } from '../../../../../platform/commands/common/commands.js';

const extensionPoint = ExtensionsRegistry.registerExtensionPoint<IChatSessionsExtensionPoint[]>({
	extensionPoint: 'chatSessions',
	jsonSchema: {
		description: localize('chatSessionsExtPoint', 'Contributes chat session integrations to the chat widget.'),
		type: 'array',
		items: {
			type: 'object',
			additionalProperties: false,
			properties: {
				type: {
					description: localize('chatSessionsExtPoint.chatSessionType', 'Unique identifier for the type of chat session.'),
					type: 'string',
				},
				name: {
					description: localize('chatSessionsExtPoint.name', 'Name of the dynamically registered chat participant (eg: @agent). Must not contain whitespace.'),
					type: 'string',
					pattern: '^[\\w-]+$'
				},
				displayName: {
					description: localize('chatSessionsExtPoint.displayName', 'A longer name for this item which is used for display in menus.'),
					type: 'string',
				},
				description: {
					description: localize('chatSessionsExtPoint.description', 'Description of the chat session for use in menus and tooltips.'),
					type: 'string'
				},
				when: {
					description: localize('chatSessionsExtPoint.when', 'Condition which must be true to show this item.'),
					type: 'string'
				},
				icon: {
					description: localize('chatSessionsExtPoint.icon', 'Icon identifier (codicon ID) for the chat session editor tab. For example, "$(github)" or "$(cloud)".'),
					anyOf: [{
						type: 'string'
					},
					{
						type: 'object',
						properties: {
							light: {
								description: localize('icon.light', 'Icon path when a light theme is used'),
								type: 'string'
							},
							dark: {
								description: localize('icon.dark', 'Icon path when a dark theme is used'),
								type: 'string'
							}
						}
					}]
				},
				order: {
					description: localize('chatSessionsExtPoint.order', 'Order in which this item should be displayed.'),
					type: 'integer'
				},
				alternativeIds: {
					description: localize('chatSessionsExtPoint.alternativeIds', 'Alternative identifiers for backward compatibility.'),
					type: 'array',
					items: {
						type: 'string'
					}
				},
				welcomeTitle: {
					description: localize('chatSessionsExtPoint.welcomeTitle', 'Title text to display in the chat welcome view for this session type.'),
					type: 'string'
				},
				welcomeMessage: {
					description: localize('chatSessionsExtPoint.welcomeMessage', 'Message text (supports markdown) to display in the chat welcome view for this session type.'),
					type: 'string'
				},
				welcomeTips: {
					description: localize('chatSessionsExtPoint.welcomeTips', 'Tips text (supports markdown and theme icons) to display in the chat welcome view for this session type.'),
					type: 'string'
				},
				inputPlaceholder: {
					description: localize('chatSessionsExtPoint.inputPlaceholder', 'Placeholder text to display in the chat input box for this session type.'),
					type: 'string'
				},
				capabilities: {
					description: localize('chatSessionsExtPoint.capabilities', 'Optional capabilities for this chat session.'),
					type: 'object',
					additionalProperties: false,
					properties: {
						supportsFileAttachments: {
							description: localize('chatSessionsExtPoint.supportsFileAttachments', 'Whether this chat session supports attaching files or file references.'),
							type: 'boolean'
						},
						supportsToolAttachments: {
							description: localize('chatSessionsExtPoint.supportsToolAttachments', 'Whether this chat session supports attaching tools or tool references.'),
							type: 'boolean'
						},
						supportsMCPAttachments: {
							description: localize('chatSessionsExtPoint.supportsMCPAttachments', 'Whether this chat session supports attaching MCP resources.'),
							type: 'boolean'
						},
						supportsImageAttachments: {
							description: localize('chatSessionsExtPoint.supportsImageAttachments', 'Whether this chat session supports attaching images.'),
							type: 'boolean'
						},
						supportsSearchResultAttachments: {
							description: localize('chatSessionsExtPoint.supportsSearchResultAttachments', 'Whether this chat session supports attaching search results.'),
							type: 'boolean'
						},
						supportsInstructionAttachments: {
							description: localize('chatSessionsExtPoint.supportsInstructionAttachments', 'Whether this chat session supports attaching instructions.'),
							type: 'boolean'
						},
						supportsSourceControlAttachments: {
							description: localize('chatSessionsExtPoint.supportsSourceControlAttachments', 'Whether this chat session supports attaching source control changes.'),
							type: 'boolean'
						},
						supportsProblemAttachments: {
							description: localize('chatSessionsExtPoint.supportsProblemAttachments', 'Whether this chat session supports attaching problems.'),
							type: 'boolean'
						},
						supportsSymbolAttachments: {
							description: localize('chatSessionsExtPoint.supportsSymbolAttachments', 'Whether this chat session supports attaching symbols.'),
							type: 'boolean'
						}
					}
				},
				commands: {
					markdownDescription: localize('chatCommandsDescription', "Commands available for this chat session, which the user can invoke with a `/`."),
					type: 'array',
					items: {
						additionalProperties: false,
						type: 'object',
						defaultSnippets: [{ body: { name: '', description: '' } }],
						required: ['name'],
						properties: {
							name: {
								description: localize('chatCommand', "A short name by which this command is referred to in the UI, e.g. `fix` or `explain` for commands that fix an issue or explain code. The name should be unique among the commands provided by this participant."),
								type: 'string'
							},
							description: {
								description: localize('chatCommandDescription', "A description of this command."),
								type: 'string'
							},
							when: {
								description: localize('chatCommandWhen', "A condition which must be true to enable this command."),
								type: 'string'
							},
						}
					}
				},
				canDelegate: {
					description: localize('chatSessionsExtPoint.canDelegate', 'Whether delegation is supported. Default is false. Note that enabling this is experimental and may not be respected at all times.'),
					type: 'boolean',
					default: false
				},
				customAgentTarget: {
					description: localize('chatSessionsExtPoint.customAgentTarget', 'When set, the chat session will show a filtered mode picker that prefers custom agents whose target property matches this value. Custom agents without a target property are still shown in all session types. This enables the use of standard agent/mode with contributed sessions.'),
					type: 'string'
				}
			},
			required: ['type', 'name', 'displayName', 'description'],
		}
	},
	activationEventsGenerator: function* (contribs) {
		for (const contrib of contribs) {
			yield `onChatSession:${contrib.type}`;
		}
	}
});

class ContributedChatSessionData extends Disposable {

	private readonly _optionsCache: Map<string /* 'models' */, string | IChatSessionProviderOptionItem>;
	public getOption(optionId: string): string | IChatSessionProviderOptionItem | undefined {
		return this._optionsCache.get(optionId);
	}
	public setOption(optionId: string, value: string | IChatSessionProviderOptionItem): void {
		this._optionsCache.set(optionId, value);
	}

	constructor(
		readonly session: IChatSession,
		readonly chatSessionType: string,
		readonly resource: URI,
		readonly options: Record<string, string | IChatSessionProviderOptionItem> | undefined,
		private readonly onWillDispose: (resource: URI) => void
	) {
		super();

		this._optionsCache = new Map<string, string | IChatSessionProviderOptionItem>();
		if (options) {
			for (const [key, value] of Object.entries(options)) {
				this._optionsCache.set(key, value);
			}
		}

		this._register(this.session.onWillDispose(() => {
			this.onWillDispose(this.resource);
		}));
	}
}


export class ChatSessionsService extends Disposable implements IChatSessionsService {
	readonly _serviceBrand: undefined;

	private readonly _itemsProviders: Map</* type */ string, IChatSessionItemProvider> = new Map();

	private readonly _contributions: Map</* type */ string, { readonly contribution: IChatSessionsExtensionPoint; readonly extension: IRelaxedExtensionDescription }> = new Map();
	private readonly _contributionDisposables = this._register(new DisposableMap</* type */ string>());

	private readonly _contentProviders: Map</* scheme */ string, IChatSessionContentProvider> = new Map();
	private readonly _alternativeIdMap: Map</* alternativeId */ string, /* primaryType */ string> = new Map();
	private readonly _contextKeys = new Set<string>();

	private readonly _onDidChangeItemsProviders = this._register(new Emitter<IChatSessionItemProvider>());
	readonly onDidChangeItemsProviders: Event<IChatSessionItemProvider> = this._onDidChangeItemsProviders.event;

	private readonly _onDidChangeSessionItems = this._register(new Emitter<string>());
	readonly onDidChangeSessionItems: Event<string> = this._onDidChangeSessionItems.event;

	private readonly _onDidChangeAvailability = this._register(new Emitter<void>());
	readonly onDidChangeAvailability: Event<void> = this._onDidChangeAvailability.event;

	private readonly _onDidChangeInProgress = this._register(new Emitter<void>());
	public get onDidChangeInProgress() { return this._onDidChangeInProgress.event; }

	private readonly _onDidChangeContentProviderSchemes = this._register(new Emitter<{ readonly added: string[]; readonly removed: string[] }>());
	public get onDidChangeContentProviderSchemes() { return this._onDidChangeContentProviderSchemes.event; }
	private readonly _onDidChangeSessionOptions = this._register(new Emitter<URI>());
	public get onDidChangeSessionOptions() { return this._onDidChangeSessionOptions.event; }
	private readonly _onDidChangeOptionGroups = this._register(new Emitter<string>());
	public get onDidChangeOptionGroups() { return this._onDidChangeOptionGroups.event; }
	private readonly _onRequestNotifyExtension = this._register(new AsyncEmitter<IChatSessionOptionsWillNotifyExtensionEvent>());
	public get onRequestNotifyExtension() { return this._onRequestNotifyExtension.event; }

	private readonly inProgressMap: Map<string, number> = new Map();
	private readonly _sessionTypeOptions: Map<string, IChatSessionProviderOptionGroup[]> = new Map();
	private readonly _sessionTypeIcons: Map<string, ThemeIcon | { light: URI; dark: URI }> = new Map();
	private readonly _sessionTypeWelcomeTitles: Map<string, string> = new Map();
	private readonly _sessionTypeWelcomeMessages: Map<string, string> = new Map();
	private readonly _sessionTypeWelcomeTips: Map<string, string> = new Map();
	private readonly _sessionTypeInputPlaceholders: Map<string, string> = new Map();

	private readonly _sessions = new ResourceMap<ContributedChatSessionData>();

	private readonly _hasCanDelegateProvidersKey: IContextKey<boolean>;

	constructor(
		@ILogService private readonly _logService: ILogService,
		@IChatAgentService private readonly _chatAgentService: IChatAgentService,
		@IExtensionService private readonly _extensionService: IExtensionService,
		@IContextKeyService private readonly _contextKeyService: IContextKeyService,
		@IMenuService private readonly _menuService: IMenuService,
		@IThemeService private readonly _themeService: IThemeService,
		@ILabelService private readonly _labelService: ILabelService
	) {
		super();

		this._hasCanDelegateProvidersKey = ChatContextKeys.hasCanDelegateProviders.bindTo(this._contextKeyService);

		this._register(extensionPoint.setHandler(extensions => {
			for (const ext of extensions) {
				if (!isProposedApiEnabled(ext.description, 'chatSessionsProvider')) {
					continue;
				}
				if (!Array.isArray(ext.value)) {
					continue;
				}
				for (const contribution of ext.value) {
					this._register(this.registerContribution(contribution, ext.description));
				}
			}
		}));

		// Listen for context changes and re-evaluate contributions
		this._register(Event.filter(this._contextKeyService.onDidChangeContext, e => e.affectsSome(this._contextKeys))(() => {
			this._evaluateAvailability();
		}));

		const builtinSessionProviders = [AgentSessionProviders.Local];
		const contributedSessionProviders = observableFromEvent(
			this.onDidChangeAvailability,
			() => Array.from(this._contributions.keys()).filter(isAgentSessionProviderType) as AgentSessionProviders[],
		).recomputeInitiallyAndOnChange(this._store);

		this._register(autorun(reader => {
			const activatedProviders = [...builtinSessionProviders, ...contributedSessionProviders.read(reader)];
			for (const provider of Object.values(AgentSessionProviders)) {
				if (activatedProviders.includes(provider)) {
					reader.store.add(registerNewSessionInPlaceAction(provider, getAgentSessionProviderName(provider)));
				}
			}
		}));

		this._register(this.onDidChangeSessionItems(chatSessionType => {
			this.updateInProgressStatus(chatSessionType).catch(error => {
				this._logService.warn(`Failed to update progress status for '${chatSessionType}':`, error);
			});
		}));

		this._register(this._labelService.registerFormatter({
			scheme: Schemas.copilotPr,
			formatting: {
				label: '${authority}${path}',
				separator: sep,
				stripPathStartingSeparator: true,
			}
		}));
	}

	public reportInProgress(chatSessionType: string, count: number): void {
		let displayName: string | undefined;

		if (chatSessionType === AgentSessionProviders.Local) {
			displayName = localize('chat.session.inProgress.local', "Local Agent");
		} else if (chatSessionType === AgentSessionProviders.Background) {
			displayName = localize('chat.session.inProgress.background', "Background Agent");
		} else if (chatSessionType === AgentSessionProviders.Cloud) {
			displayName = localize('chat.session.inProgress.cloud', "Cloud Agent");
		} else {
			displayName = this._contributions.get(chatSessionType)?.contribution.displayName;
		}

		if (displayName) {
			this.inProgressMap.set(displayName, count);
		}
		this._onDidChangeInProgress.fire();
	}

	public getInProgress(): { displayName: string; count: number }[] {
		return Array.from(this.inProgressMap.entries()).map(([displayName, count]) => ({ displayName, count }));
	}

	private async updateInProgressStatus(chatSessionType: string): Promise<void> {
		try {
			const results = await this.getChatSessionItems([chatSessionType], CancellationToken.None);
			const items = results.flatMap(r => r.items);
			const inProgress = items.filter(item => item.status && isSessionInProgressStatus(item.status));
			this.reportInProgress(chatSessionType, inProgress.length);
		} catch (error) {
			this._logService.warn(`Failed to update in-progress status for chat session type '${chatSessionType}':`, error);
		}
	}

	private registerContribution(contribution: IChatSessionsExtensionPoint, ext: IRelaxedExtensionDescription): IDisposable {
		if (this._contributions.has(contribution.type)) {
			return { dispose: () => { } };
		}

		// Track context keys from the when condition
		if (contribution.when) {
			const whenExpr = ContextKeyExpr.deserialize(contribution.when);
			if (whenExpr) {
				for (const key of whenExpr.keys()) {
					this._contextKeys.add(key);
				}
			}
		}

		this._contributions.set(contribution.type, { contribution, extension: ext });

		// Register alternative IDs if provided
		if (contribution.alternativeIds) {
			for (const altId of contribution.alternativeIds) {
				if (this._alternativeIdMap.has(altId)) {
					this._logService.warn(`Alternative ID '${altId}' is already mapped to '${this._alternativeIdMap.get(altId)}'. Remapping to '${contribution.type}'.`);
				}
				this._alternativeIdMap.set(altId, contribution.type);
			}
		}

		// Store icon mapping if provided
		let icon: ThemeIcon | { dark: URI; light: URI } | undefined;

		if (contribution.icon) {
			// Parse icon string - support ThemeIcon format or file path from extension
			if (typeof contribution.icon === 'string') {
				icon = contribution.icon.startsWith('$(') && contribution.icon.endsWith(')')
					? ThemeIcon.fromString(contribution.icon)
					: ThemeIcon.fromId(contribution.icon);
			} else {
				icon = {
					dark: resources.joinPath(ext.extensionLocation, contribution.icon.dark),
					light: resources.joinPath(ext.extensionLocation, contribution.icon.light)
				};
			}
		}

		if (icon) {
			this._sessionTypeIcons.set(contribution.type, icon);
		}

		// Store welcome title, message, tips, and input placeholder if provided
		if (contribution.welcomeTitle) {
			this._sessionTypeWelcomeTitles.set(contribution.type, contribution.welcomeTitle);
		}
		if (contribution.welcomeMessage) {
			this._sessionTypeWelcomeMessages.set(contribution.type, contribution.welcomeMessage);
		}
		if (contribution.welcomeTips) {
			this._sessionTypeWelcomeTips.set(contribution.type, contribution.welcomeTips);
		}
		if (contribution.inputPlaceholder) {
			this._sessionTypeInputPlaceholders.set(contribution.type, contribution.inputPlaceholder);
		}

		this._evaluateAvailability();

		return {
			dispose: () => {
				this._contributions.delete(contribution.type);
				// Remove alternative ID mappings
				if (contribution.alternativeIds) {
					for (const altId of contribution.alternativeIds) {
						if (this._alternativeIdMap.get(altId) === contribution.type) {
							this._alternativeIdMap.delete(altId);
						}
					}
				}
				this._sessionTypeIcons.delete(contribution.type);
				this._sessionTypeWelcomeTitles.delete(contribution.type);
				this._sessionTypeWelcomeMessages.delete(contribution.type);
				this._sessionTypeWelcomeTips.delete(contribution.type);
				this._sessionTypeInputPlaceholders.delete(contribution.type);
				this._contributionDisposables.deleteAndDispose(contribution.type);
				this._updateHasCanDelegateProvidersContextKey();
			}
		};
	}

	private _isContributionAvailable(contribution: IChatSessionsExtensionPoint): boolean {
		if (!contribution.when) {
			return true;
		}
		const whenExpr = ContextKeyExpr.deserialize(contribution.when);
		return !whenExpr || this._contextKeyService.contextMatchesRules(whenExpr);
	}

	/**
	 * Resolves a session type to its primary type, checking for alternative IDs.
	 * @param sessionType The session type or alternative ID to resolve
	 * @returns The primary session type, or undefined if not found or not available
	 */
	private _resolveToPrimaryType(sessionType: string): string | undefined {
		// Try to find the primary type first
		const contribution = this._contributions.get(sessionType)?.contribution;
		if (contribution) {
			// If the contribution is available, use it
			if (this._isContributionAvailable(contribution)) {
				return sessionType;
			}
			// If not available, fall through to check for alternatives
		}

		// Check if this is an alternative ID, or if the primary type is not available
		const primaryType = this._alternativeIdMap.get(sessionType);
		if (primaryType) {
			const altContribution = this._contributions.get(primaryType)?.contribution;
			if (altContribution && this._isContributionAvailable(altContribution)) {
				return primaryType;
			}
		}

		return undefined;
	}

	private _registerMenuItems(contribution: IChatSessionsExtensionPoint, extensionDescription: IRelaxedExtensionDescription): IDisposable {
		// If provider registers anything for the create submenu, let it fully control the creation
		const contextKeyService = this._contextKeyService.createOverlay([
			['chatSessionType', contribution.type]
		]);

		const rawMenuActions = this._menuService.getMenuActions(MenuId.AgentSessionsCreateSubMenu, contextKeyService);
		const menuActions = rawMenuActions.map(value => value[1]).flat();

		const disposables = new DisposableStore();

		// Mirror all create submenu actions into the global Chat New menu
		for (let i = 0; i < menuActions.length; i++) {
			const action = menuActions[i];
			if (action instanceof MenuItemAction) {
				// TODO: This is an odd way to do this, but the best we can do currently
				if (i === 0 && !contribution.canDelegate) {
					disposables.add(registerNewSessionExternalAction(contribution.type, contribution.displayName, action.item.id));
				} else {
					disposables.add(MenuRegistry.appendMenuItem(MenuId.ChatNewMenu, {
						command: action.item,
						group: '4_externally_contributed',
					}));
				}
			}
		}
		return {
			dispose: () => disposables.dispose()
		};
	}

	private _registerCommands(contribution: IChatSessionsExtensionPoint): IDisposable {
		const isAvailableInSessionTypePicker = isAgentSessionProviderType(contribution.type);

		return combinedDisposable(
			registerAction2(class OpenChatSessionAction extends Action2 {
				constructor() {
					super({
						id: `workbench.action.chat.openSessionWithPrompt.${contribution.type}`,
						title: localize2('interactiveSession.openSessionWithPrompt', "New {0} with Prompt", contribution.displayName),
						category: CHAT_CATEGORY,
						icon: Codicon.plus,
						f1: false,
						precondition: ChatContextKeys.enabled
					});
				}

				async run(accessor: ServicesAccessor, chatOptions?: { resource: UriComponents; prompt: string; attachedContext?: IChatRequestVariableEntry[] }): Promise<void> {
					const chatService = accessor.get(IChatService);
					const { type } = contribution;

					if (chatOptions) {
						const resource = URI.revive(chatOptions.resource);
						const ref = await chatService.loadSessionForResource(resource, ChatAgentLocation.Chat, CancellationToken.None);
						await chatService.sendRequest(resource, chatOptions.prompt, { agentIdSilent: type, attachedContext: chatOptions.attachedContext });
						ref?.dispose();
					}
				}
			}),
			// Creates a chat editor
			registerAction2(class OpenNewChatSessionEditorAction extends Action2 {
				constructor() {
					super({
						id: `workbench.action.chat.openNewSessionEditor.${contribution.type}`,
						title: localize2('interactiveSession.openNewSessionEditor', "New {0}", contribution.displayName),
						category: CHAT_CATEGORY,
						icon: Codicon.plus,
						f1: true,
						precondition: ChatContextKeys.enabled,
					});
				}

				async run(accessor: ServicesAccessor, chatOptions?: { prompt: string; attachedContext?: IChatRequestVariableEntry[] }): Promise<void> {
					const { type, displayName } = contribution;
					await openChatSession(accessor, { type, displayName, position: ChatSessionPosition.Editor }, chatOptions);
				}
			}),
			// New chat in sidebar chat (+ button)
			registerAction2(class OpenNewChatSessionSidebarAction extends Action2 {
				constructor() {
					super({
						id: `workbench.action.chat.openNewSessionSidebar.${contribution.type}`,
						title: localize2('interactiveSession.openNewSessionSidebar', "New {0}", contribution.displayName),
						category: CHAT_CATEGORY,
						icon: Codicon.plus,
						f1: false, // Hide from Command Palette
						precondition: ChatContextKeys.enabled,
						menu: !isAvailableInSessionTypePicker ? {
							id: MenuId.ChatNewMenu,
							group: '3_new_special',
						} : undefined,
					});
				}

				async run(accessor: ServicesAccessor, chatOptions?: { prompt: string; attachedContext?: IChatRequestVariableEntry[] }): Promise<void> {
					const { type, displayName } = contribution;
					await openChatSession(accessor, { type, displayName, position: ChatSessionPosition.Sidebar }, chatOptions);
				}
			})
		);
	}

	private _evaluateAvailability(): void {
		let hasChanges = false;
		for (const { contribution, extension } of this._contributions.values()) {
			const isCurrentlyRegistered = this._contributionDisposables.has(contribution.type);
			const shouldBeRegistered = this._isContributionAvailable(contribution);
			if (isCurrentlyRegistered && !shouldBeRegistered) {
				// Disable the contribution by disposing its disposable store
				this._contributionDisposables.deleteAndDispose(contribution.type);

				// Also dispose any cached sessions for this contribution
				this._disposeSessionsForContribution(contribution.type);
				hasChanges = true;
			} else if (!isCurrentlyRegistered && shouldBeRegistered) {
				// Enable the contribution by registering it
				this._enableContribution(contribution, extension);
				hasChanges = true;
			}
		}
		if (hasChanges) {
			this._onDidChangeAvailability.fire();
			for (const provider of this._itemsProviders.values()) {
				this._onDidChangeItemsProviders.fire(provider);
			}
			for (const { contribution } of this._contributions.values()) {
				this._onDidChangeSessionItems.fire(contribution.type);
			}
		}
		this._updateHasCanDelegateProvidersContextKey();
	}

	private _enableContribution(contribution: IChatSessionsExtensionPoint, ext: IRelaxedExtensionDescription): void {
		const disposableStore = new DisposableStore();
		this._contributionDisposables.set(contribution.type, disposableStore);
		if (contribution.canDelegate) {
			disposableStore.add(this._registerAgent(contribution, ext));
			disposableStore.add(this._registerCommands(contribution));
		}
		disposableStore.add(this._registerMenuItems(contribution, ext));
	}

	private _disposeSessionsForContribution(contributionId: string): void {
		// Find and dispose all sessions that belong to this contribution
		const sessionsToDispose: URI[] = [];
		for (const [sessionResource, sessionData] of this._sessions) {
			if (sessionData.chatSessionType === contributionId) {
				sessionsToDispose.push(sessionResource);
			}
		}

		if (sessionsToDispose.length > 0) {
			this._logService.info(`Disposing ${sessionsToDispose.length} cached sessions for contribution '${contributionId}' due to when clause change`);
		}

		for (const sessionKey of sessionsToDispose) {
			const sessionData = this._sessions.get(sessionKey);
			if (sessionData) {
				sessionData.dispose(); // This will call _onWillDisposeSession and clean up
			}
		}
	}

	private _registerAgent(contribution: IChatSessionsExtensionPoint, ext: IRelaxedExtensionDescription): IDisposable {
		const { type: id, name, displayName, description } = contribution;
		const storedIcon = this._sessionTypeIcons.get(id);
		const icons = ThemeIcon.isThemeIcon(storedIcon)
			? { themeIcon: storedIcon, icon: undefined, iconDark: undefined }
			: storedIcon
				? { icon: storedIcon.light, iconDark: storedIcon.dark }
				: { themeIcon: Codicon.sendToRemoteAgent };

		const agentData: IChatAgentData = {
			id,
			name,
			fullName: displayName,
			description: description,
			isDefault: false,
			isCore: false,
			isDynamic: true,
			slashCommands: contribution.commands ?? [],
			locations: [ChatAgentLocation.Chat],
			modes: [ChatModeKind.Agent, ChatModeKind.Ask],
			disambiguation: [],
			metadata: {
				...icons,
			},
			capabilities: contribution.capabilities,
			canAccessPreviousChatHistory: true,
			extensionId: ext.identifier,
			extensionVersion: ext.version,
			extensionDisplayName: ext.displayName || ext.name,
			extensionPublisherId: ext.publisher,
		};

		return this._chatAgentService.registerAgent(id, agentData);
	}

	getAllChatSessionContributions(): IChatSessionsExtensionPoint[] {
		return Array.from(this._contributions.values(), x => x.contribution)
			.filter(contribution => this._isContributionAvailable(contribution));
	}

	private _updateHasCanDelegateProvidersContextKey(): void {
		const hasCanDelegate = this.getAllChatSessionContributions().filter(c => c.canDelegate);
		const canDelegateEnabled = hasCanDelegate.length > 0;
		this._logService.trace(`[ChatSessionsService] hasCanDelegateProvidersAvailable=${canDelegateEnabled} (${hasCanDelegate.map(c => c.type).join(', ')})`);
		this._hasCanDelegateProvidersKey.set(canDelegateEnabled);
	}

	getChatSessionContribution(chatSessionType: string): IChatSessionsExtensionPoint | undefined {
		const contribution = this._contributions.get(chatSessionType)?.contribution;
		if (!contribution) {
			return undefined;
		}

		return this._isContributionAvailable(contribution) ? contribution : undefined;
	}

	async activateChatSessionItemProvider(chatViewType: string): Promise<IChatSessionItemProvider | undefined> {
		await this._extensionService.whenInstalledExtensionsRegistered();
		const resolvedType = this._resolveToPrimaryType(chatViewType);
		if (resolvedType) {
			chatViewType = resolvedType;
		}

		const contribution = this._contributions.get(chatViewType)?.contribution;
		if (contribution && !this._isContributionAvailable(contribution)) {
			return undefined;
		}

		if (this._itemsProviders.has(chatViewType)) {
			return this._itemsProviders.get(chatViewType);
		}

		await this._extensionService.activateByEvent(`onChatSession:${chatViewType}`);

		return this._itemsProviders.get(chatViewType);
	}

	async canResolveChatSession(chatSessionResource: URI) {
		await this._extensionService.whenInstalledExtensionsRegistered();
		const resolvedType = this._resolveToPrimaryType(chatSessionResource.scheme) || chatSessionResource.scheme;
		const contribution = this._contributions.get(resolvedType)?.contribution;
		if (contribution && !this._isContributionAvailable(contribution)) {
			return false;
		}

		if (this._contentProviders.has(chatSessionResource.scheme)) {
			return true;
		}

		await this._extensionService.activateByEvent(`onChatSession:${chatSessionResource.scheme}`);
		return this._contentProviders.has(chatSessionResource.scheme);
	}

	public async getChatSessionItems(providersToResolve: readonly string[] | undefined, token: CancellationToken): Promise<Array<{ readonly chatSessionType: string; readonly items: IChatSessionItem[] }>> {
		const results: Array<{ readonly chatSessionType: string; readonly items: IChatSessionItem[] }> = [];
		const resolvedProviderTypes = new Set<string>();

		// First, iterate over extension point contributions
		for (const contrib of this.getAllChatSessionContributions()) {
			if (providersToResolve && !providersToResolve.includes(contrib.type)) {
				continue; // skip: not considered for resolving
			}

			const provider = await this.activateChatSessionItemProvider(contrib.type);
			if (!provider) {
				// We requested this provider but it is not available
				if (providersToResolve?.includes(contrib.type)) {
					this._logService.trace(`[ChatSessionsService] No enabled provider found for chat session type ${contrib.type}`);
				}
				continue;
			}

			try {
				const providerSessions = await raceCancellationError(provider.provideChatSessionItems(token), token);
				this._logService.trace(`[ChatSessionsService] Resolved ${providerSessions.length} sessions for provider ${provider.chatSessionType}`);
				results.push({ chatSessionType: provider.chatSessionType, items: providerSessions });
				resolvedProviderTypes.add(provider.chatSessionType);
			} catch (error) {
				// Log error but continue with other providers
				this._logService.error(`[ChatSessionsService] Failed to resolve sessions for provider ${provider.chatSessionType}`, error);
				continue;
			}
		}

		// Also include registered items providers that don't have corresponding contributions
		// (e.g., the local session provider which is built-in and not an extension contribution)
		for (const [chatSessionType, provider] of this._itemsProviders) {
			if (resolvedProviderTypes.has(chatSessionType)) {
				continue; // already resolved via contribution
			}
			if (providersToResolve && !providersToResolve.includes(chatSessionType)) {
				continue; // skip: not considered for resolving
			}

			try {
				const providerSessions = await raceCancellationError(provider.provideChatSessionItems(token), token);
				this._logService.trace(`[ChatSessionsService] Resolved ${providerSessions.length} sessions for built-in provider ${chatSessionType}`);
				results.push({ chatSessionType, items: providerSessions });
			} catch (error) {
				this._logService.error(`[ChatSessionsService] Failed to resolve sessions for built-in provider ${chatSessionType}`, error);
				continue;
			}
		}

		return results;
	}

	public registerChatSessionItemProvider(provider: IChatSessionItemProvider): IDisposable {
		const chatSessionType = provider.chatSessionType;
		this._itemsProviders.set(chatSessionType, provider);
		this._onDidChangeItemsProviders.fire(provider);

		const disposables = new DisposableStore();
		disposables.add(provider.onDidChangeChatSessionItems(() => {
			this._onDidChangeSessionItems.fire(chatSessionType);
		}));

		this.updateInProgressStatus(chatSessionType).catch(error => {
			this._logService.warn(`Failed to update initial progress status for '${chatSessionType}':`, error);
		});

		return {
			dispose: () => {
				disposables.dispose();

				const provider = this._itemsProviders.get(chatSessionType);
				if (provider) {
					this._itemsProviders.delete(chatSessionType);
					this._onDidChangeItemsProviders.fire(provider);
				}
			}
		};
	}

	registerChatSessionContentProvider(chatSessionType: string, provider: IChatSessionContentProvider): IDisposable {
		if (this._contentProviders.has(chatSessionType)) {
			throw new Error(`Content provider for ${chatSessionType} is already registered.`);
		}

		this._contentProviders.set(chatSessionType, provider);
		this._onDidChangeContentProviderSchemes.fire({ added: [chatSessionType], removed: [] });

		return {
			dispose: () => {
				this._contentProviders.delete(chatSessionType);

				this._onDidChangeContentProviderSchemes.fire({ added: [], removed: [chatSessionType] });

				// Remove all sessions that were created by this provider
				for (const [key, session] of this._sessions) {
					if (session.chatSessionType === chatSessionType) {
						session.dispose();
						this._sessions.delete(key);
					}
				}
			}
		};
	}

	public registerChatModelChangeListeners(
		chatService: IChatService,
		chatSessionType: string,
		onChange: () => void
	): IDisposable {
		const disposableStore = new DisposableStore();
		const chatModelsICareAbout = chatService.chatModels.map(models =>
			Array.from(models).filter((model: IChatModel) => model.sessionResource.scheme === chatSessionType)
		);

		const listeners = new ResourceMap<IDisposable>();
		const autoRunDisposable = autorunIterableDelta(
			reader => chatModelsICareAbout.read(reader),
			({ addedValues, removedValues }) => {
				removedValues.forEach((removed) => {
					const listener = listeners.get(removed.sessionResource);
					if (listener) {
						listeners.delete(removed.sessionResource);
						listener.dispose();
					}
				});
				addedValues.forEach((added) => {
					const requestChangeListener = added.lastRequestObs.map(last => last?.response && observableSignalFromEvent('chatSessions.modelRequestChangeListener', last.response.onDidChange));
					const modelChangeListener = observableSignalFromEvent('chatSessions.modelChangeListener', added.onDidChange);
					listeners.set(added.sessionResource, autorun(reader => {
						requestChangeListener.read(reader)?.read(reader);
						modelChangeListener.read(reader);
						onChange();
					}));
				});
			}
		);
		disposableStore.add(toDisposable(() => {
			for (const listener of listeners.values()) { listener.dispose(); }
		}));
		disposableStore.add(autoRunDisposable);
		return disposableStore;
	}


	public getInProgressSessionDescription(chatModel: IChatModel): string | undefined {
		const requests = chatModel.getRequests();
		if (requests.length === 0) {
			return undefined;
		}

		// Get the last request to check its response status
		const lastRequest = requests.at(-1);
		const response = lastRequest?.response;
		if (!response) {
			return undefined;
		}

		// If the response is complete, show Finished
		if (response.isComplete) {
			return undefined;
		}

		// Get the response parts to find tool invocations and progress messages
		const responseParts = response.response.value;
		let description: string | IMarkdownString | undefined = '';

		for (let i = responseParts.length - 1; i >= 0; i--) {
			const part = responseParts[i];
			if (description) {
				break;
			}

			if (part.kind === 'confirmation' && typeof part.message === 'string') {
				description = part.message;
			} else if (part.kind === 'toolInvocation') {
				const toolInvocation = part as IChatToolInvocation;
				const state = toolInvocation.state.get();
				description = toolInvocation.generatedTitle || toolInvocation.pastTenseMessage || toolInvocation.invocationMessage;
				if (state.type === IChatToolInvocation.StateKind.WaitingForConfirmation) {
					const confirmationTitle = state.confirmationMessages?.title;
					const titleMessage = confirmationTitle && (typeof confirmationTitle === 'string'
						? confirmationTitle
						: confirmationTitle.value);
					const descriptionValue = typeof description === 'string' ? description : description.value;
					description = titleMessage ?? localize('chat.sessions.description.waitingForConfirmation', "Waiting for confirmation: {0}", descriptionValue);
				}
			} else if (part.kind === 'toolInvocationSerialized') {
				description = part.invocationMessage;
			} else if (part.kind === 'progressMessage') {
				description = part.content;
			} else if (part.kind === 'thinking') {
				description = localize('chat.sessions.description.thinking', 'Thinking...');
			}
		}

		return description ? renderAsPlaintext(description, { useLinkFormatter: true }) : '';
	}

	public async getOrCreateChatSession(sessionResource: URI, token: CancellationToken): Promise<IChatSession> {
		const existingSessionData = this._sessions.get(sessionResource);
		if (existingSessionData) {
			return existingSessionData.session;
		}

		if (!(await raceCancellationError(this.canResolveChatSession(sessionResource), token))) {
			throw Error(`Can not find provider for ${sessionResource}`);
		}

		const resolvedType = this._resolveToPrimaryType(sessionResource.scheme) || sessionResource.scheme;
		const provider = this._contentProviders.get(resolvedType);
		if (!provider) {
			throw Error(`Can not find provider for ${sessionResource}`);
		}

		const session = await raceCancellationError(provider.provideChatSessionContent(sessionResource, token), token);
		const sessionData = new ContributedChatSessionData(session, sessionResource.scheme, sessionResource, session.options, resource => {
			sessionData.dispose();
			this._sessions.delete(resource);
		});

		this._sessions.set(sessionResource, sessionData);

		return session;
	}

	public hasAnySessionOptions(sessionResource: URI): boolean {
		const session = this._sessions.get(sessionResource);
		return !!session && !!session.options && Object.keys(session.options).length > 0;
	}

	public getSessionOption(sessionResource: URI, optionId: string): string | IChatSessionProviderOptionItem | undefined {
		const session = this._sessions.get(sessionResource);
		return session?.getOption(optionId);
	}

	public setSessionOption(sessionResource: URI, optionId: string, value: string | IChatSessionProviderOptionItem): boolean {
		const session = this._sessions.get(sessionResource);
		return !!session?.setOption(optionId, value);
	}

	public notifySessionItemsChanged(chatSessionType: string): void {
		this._onDidChangeSessionItems.fire(chatSessionType);
	}

	/**
	 * Store option groups for a session type
	 */
	public setOptionGroupsForSessionType(chatSessionType: string, handle: number, optionGroups?: IChatSessionProviderOptionGroup[]): void {
		if (optionGroups) {
			this._sessionTypeOptions.set(chatSessionType, optionGroups);
		} else {
			this._sessionTypeOptions.delete(chatSessionType);
		}
		this._onDidChangeOptionGroups.fire(chatSessionType);
	}

	/**
	 * Get available option groups for a session type
	 */
	public getOptionGroupsForSessionType(chatSessionType: string): IChatSessionProviderOptionGroup[] | undefined {
		return this._sessionTypeOptions.get(chatSessionType);
	}

	/**
	 * Notify extension about option changes for a session
	 */
	public async notifySessionOptionsChange(sessionResource: URI, updates: ReadonlyArray<{ optionId: string; value: string | IChatSessionProviderOptionItem }>): Promise<void> {
		if (!updates.length) {
			return;
		}
		this._logService.trace(`[ChatSessionsService] notifySessionOptionsChange: starting for ${sessionResource}, ${updates.length} update(s): [${updates.map(u => u.optionId).join(', ')}]`);
		// Fire event to notify MainThreadChatSessions (which forwards to extension host)
		// Uses fireAsync to properly await async listener work via waitUntil pattern
		await this._onRequestNotifyExtension.fireAsync({ sessionResource, updates }, CancellationToken.None);
		this._logService.trace(`[ChatSessionsService] notifySessionOptionsChange: fireAsync completed for ${sessionResource}`);
		for (const u of updates) {
			this.setSessionOption(sessionResource, u.optionId, u.value);
		}
		this._onDidChangeSessionOptions.fire(sessionResource);
		this._logService.trace(`[ChatSessionsService] notifySessionOptionsChange: finished for ${sessionResource}`);
	}

	/**
	 * Get the icon for a specific session type
	 */
	public getIconForSessionType(chatSessionType: string): ThemeIcon | URI | undefined {
		const sessionTypeIcon = this._sessionTypeIcons.get(chatSessionType);

		if (ThemeIcon.isThemeIcon(sessionTypeIcon)) {
			return sessionTypeIcon;
		}

		if (isDark(this._themeService.getColorTheme().type)) {
			return sessionTypeIcon?.dark;
		} else {
			return sessionTypeIcon?.light;
		}
	}

	/**
	 * Get the welcome title for a specific session type
	 */
	public getWelcomeTitleForSessionType(chatSessionType: string): string | undefined {
		return this._sessionTypeWelcomeTitles.get(chatSessionType);
	}

	/**
	 * Get the welcome message for a specific session type
	 */
	public getWelcomeMessageForSessionType(chatSessionType: string): string | undefined {
		return this._sessionTypeWelcomeMessages.get(chatSessionType);
	}

	/**
	 * Get the input placeholder for a specific session type
	 */
	public getInputPlaceholderForSessionType(chatSessionType: string): string | undefined {
		return this._sessionTypeInputPlaceholders.get(chatSessionType);
	}

	/**
	 * Get the capabilities for a specific session type
	 */
	public getCapabilitiesForSessionType(chatSessionType: string): IChatAgentAttachmentCapabilities | undefined {
		const contribution = this._contributions.get(chatSessionType)?.contribution;
		return contribution?.capabilities;
	}

	/**
	 * Get the customAgentTarget for a specific session type.
	 * When set, the mode picker should show filtered custom agents matching this target.
	 */
	public getCustomAgentTargetForSessionType(chatSessionType: string): string | undefined {
		const contribution = this._contributions.get(chatSessionType)?.contribution;
		return contribution?.customAgentTarget;
	}

	public getContentProviderSchemes(): string[] {
		return Array.from(this._contentProviders.keys());
	}
}

registerSingleton(IChatSessionsService, ChatSessionsService, InstantiationType.Delayed);

function registerNewSessionInPlaceAction(type: string, displayName: string): IDisposable {
	return registerAction2(class NewChatSessionInPlaceAction extends Action2 {
		constructor() {
			super({
				id: `workbench.action.chat.openNewChatSessionInPlace.${type}`,
				title: localize2('interactiveSession.openNewChatSessionInPlace', "New {0}", displayName),
				category: CHAT_CATEGORY,
				f1: false,
				precondition: ChatContextKeys.enabled,
			});
		}

		// Expected args: [chatSessionPosition: 'sidebar' | 'editor']
		async run(accessor: ServicesAccessor, ...args: unknown[]): Promise<void> {
			if (args.length === 0) {
				throw new BugIndicatingError('Expected chat session position argument');
			}

			const chatSessionPosition = args[0];
			if (chatSessionPosition !== ChatSessionPosition.Sidebar && chatSessionPosition !== ChatSessionPosition.Editor) {
				throw new BugIndicatingError(`Invalid chat session position argument: ${chatSessionPosition}`);
			}

			await openChatSession(accessor, { type: type, displayName: localize('chat', "Chat"), position: chatSessionPosition, replaceEditor: true });
		}
	});
}

function registerNewSessionExternalAction(type: string, displayName: string, commandId: string): IDisposable {
	return registerAction2(class NewChatSessionExternalAction extends Action2 {
		constructor() {
			super({
				id: `workbench.action.chat.openNewChatSessionExternal.${type}`,
				title: localize2('interactiveSession.openNewChatSessionExternal', "New {0}", displayName),
				category: CHAT_CATEGORY,
				f1: false,
				precondition: ChatContextKeys.enabled,
			});
		}
		async run(accessor: ServicesAccessor): Promise<void> {
			const commandService = accessor.get(ICommandService);
			await commandService.executeCommand(commandId);
		}
	});
}

export enum ChatSessionPosition {
	Editor = 'editor',
	Sidebar = 'sidebar'
}

type NewChatSessionSendOptions = {
	readonly prompt: string;
	readonly attachedContext?: IChatRequestVariableEntry[];
};

export type NewChatSessionOpenOptions = {
	readonly type: string;
	readonly position: ChatSessionPosition;
	readonly displayName: string;
	readonly chatResource?: UriComponents;
	readonly replaceEditor?: boolean;
};

async function openChatSession(accessor: ServicesAccessor, openOptions: NewChatSessionOpenOptions, chatSendOptions?: NewChatSessionSendOptions): Promise<void> {
	const viewsService = accessor.get(IViewsService);
	const chatService = accessor.get(IChatService);
	const logService = accessor.get(ILogService);
	const editorGroupService = accessor.get(IEditorGroupsService);
	const editorService = accessor.get(IEditorService);

	// Determine resource to open
	const resource = getResourceForNewChatSession(openOptions);

	// Open chat session
	try {
		switch (openOptions.position) {
			case ChatSessionPosition.Sidebar: {
				const view = await viewsService.openView(ChatViewId) as ChatViewPane;
				if (openOptions.type === AgentSessionProviders.Local) {
					await view.widget.clear();
				} else {
					await view.loadSession(resource);
				}
				view.focus();
				break;
			}
			case ChatSessionPosition.Editor: {
				const options: IChatEditorOptions = {
					override: ChatEditorInput.EditorID,
					pinned: true,
					title: {
						fallback: localize('chatEditorContributionName', "{0}", openOptions.displayName),
					}
				};
				if (openOptions.replaceEditor) {
					// TODO: Do not rely on active editor
					const activeEditor = editorGroupService.activeGroup.activeEditor;
					if (!activeEditor || !(activeEditor instanceof ChatEditorInput)) {
						throw new Error('No active chat editor to replace');
					}
					await editorService.replaceEditors([{ editor: activeEditor, replacement: { resource, options } }], editorGroupService.activeGroup);
				} else {
					await editorService.openEditor({ resource, options });
				}
				break;
			}
			default: assertNever(openOptions.position, `Unknown chat session position: ${openOptions.position}`);
		}
	} catch (e) {
		logService.error(`Failed to open '${openOptions.type}' chat session with openOptions: ${JSON.stringify(openOptions)}`, e);
		return;
	}

	// Send initial prompt if provided
	if (chatSendOptions) {
		try {
			await chatService.sendRequest(resource, chatSendOptions.prompt, { agentIdSilent: openOptions.type, attachedContext: chatSendOptions.attachedContext });
		} catch (e) {
			logService.error(`Failed to send initial request to '${openOptions.type}' chat session with contextOptions: ${JSON.stringify(chatSendOptions)}`, e);
		}
	}
}

export function getResourceForNewChatSession(options: NewChatSessionOpenOptions): URI {
	if (options.chatResource) {
		return URI.revive(options.chatResource);
	}

	const isRemoteSession = options.type !== AgentSessionProviders.Local;
	if (isRemoteSession) {
		return URI.from({
			scheme: options.type,
			path: `/untitled-${generateUuid()}`,
		});
	}

	const isEditorPosition = options.position === ChatSessionPosition.Editor;
	if (isEditorPosition) {
		return ChatEditorInput.getNewEditorUri();
	}

	return LocalChatSessionUri.forSession(generateUuid());
}

function isAgentSessionProviderType(type: string): boolean {
	return Object.values(AgentSessionProviders).includes(type as AgentSessionProviders);
}
